<?php
// reviews.php - Страница отзывов админ-панели
session_start();
if (!isset($_SESSION['loggedin']) || $_SESSION['loggedin'] !== true) {
    header("Location: login.php");
    exit();
}

// Функция для получения всех отзывов
function getAllReviews() {
    $reviews = [];
    $otziv_dir = 'otziv';
    
    if (!is_dir($otziv_dir)) {
        return $reviews;
    }
    
    $user_dirs = scandir($otziv_dir);
    
    foreach ($user_dirs as $user_id) {
        if ($user_id === '.' || $user_id === '..') continue;
        
        $user_path = $otziv_dir . '/' . $user_id;
        if (!is_dir($user_path)) continue;
        
        $files = scandir($user_path);
        
        foreach ($files as $file) {
            if (pathinfo($file, PATHINFO_EXTENSION) === 'json') {
                $file_path = $user_path . '/' . $file;
                $content = file_get_contents($file_path);
                $review_data = json_decode($content, true);
                
                if ($review_data) {
                    $review_data['user_id'] = $user_id;
                    $review_data['file_name'] = $file;
                    $reviews[] = $review_data;
                }
            }
        }
    }
    
    // Сортируем по timestamp в обратном порядке (новые сверху)
    usort($reviews, function($a, $b) {
        return strtotime($b['timestamp'] ?? '2000-01-01') - strtotime($a['timestamp'] ?? '2000-01-01');
    });
    
    return $reviews;
}

// Обработка удаления отзыва
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete') {
    $user_id = $_POST['user_id'] ?? '';
    $file_name = $_POST['file_name'] ?? '';
    $file_path = 'otziv/' . $user_id . '/' . $file_name;
    
    if (file_exists($file_path)) {
        unlink($file_path);
        header("Location: reviews.php");
        exit();
    }
}

$all_reviews = getAllReviews();
?>
<!DOCTYPE html>
<html lang="ru">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Отзывы - Админ-панель</title>
    <script src="https://cdn.tailwindcss.com"></script>
</head>
<body class="bg-gradient-to-br from-slate-50 to-blue-50 min-h-screen flex">
    <!-- Sidebar -->
    <?php include 'admin_sidebar.php'; ?>

    <!-- Main Content -->
    <div class="flex-1 lg:ml-64 p-6">
        <div class="max-w-4xl mx-auto">
            <!-- Header -->
            <div class="mb-8">
                <h1 class="text-4xl font-bold text-gray-800 mb-2">⭐ Отзывы</h1>
                <p class="text-gray-600">Всего отзывов: <span class="font-semibold text-blue-600"><?php echo count($all_reviews); ?></span></p>
            </div>

            <!-- Отзывы -->
            <?php if (empty($all_reviews)): ?>
                <div class="bg-white rounded-2xl shadow-lg p-8 text-center">
                    <p class="text-gray-500 text-lg">Отзывов пока нет</p>
                </div>
            <?php else: ?>
                <div class="space-y-4">
                    <?php foreach ($all_reviews as $review): ?>
                        <?php
                            $rating = $review['reyting'] ?? 0;
                            $text = $review['text'] ?? '';
                            $timestamp = $review['timestamp'] ?? '';
                            $user_id = $review['user_id'] ?? '';
                        ?>
                        <div class="bg-white rounded-2xl shadow-lg p-6 hover:shadow-xl transition-shadow">
                            <!-- Заголовок отзыва -->
                            <div class="flex justify-between items-start mb-4">
                                <div class="flex-1">
                                    <!-- Рейтинг (звёзды) -->
                                    <div class="flex items-center gap-2 mb-2">
                                        <div class="flex gap-1">
                                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                                <span class="text-xl <?php echo $i <= $rating ? 'text-yellow-400' : 'text-gray-300'; ?>">★</span>
                                            <?php endfor; ?>
                                        </div>
                                        <span class="text-sm font-semibold text-gray-700"><?php echo $rating; ?>/5</span>
                                    </div>
                                    
                                    <!-- ID пользователя и время -->
                                    <div class="text-sm text-gray-500">
                                        <span>👤 ID: <?php echo htmlspecialchars($user_id); ?></span>
                                        <span class="mx-2">•</span>
                                        <span>🕐 <?php echo htmlspecialchars($timestamp); ?></span>
                                    </div>
                                </div>
                                
                                <!-- Кнопка удаления -->
                                <form method="POST" class="ml-4" onsubmit="return confirm('Вы уверены, что хотите удалить этот отзыв?');">
                                    <input type="hidden" name="action" value="delete">
                                    <input type="hidden" name="user_id" value="<?php echo htmlspecialchars($user_id); ?>">
                                    <input type="hidden" name="file_name" value="<?php echo htmlspecialchars($review['file_name']); ?>">
                                    <button type="submit" class="bg-red-500 hover:bg-red-600 text-white px-4 py-2 rounded-lg transition text-sm font-medium">
                                        🗑️ Удалить
                                    </button>
                                </form>
                            </div>

                            <!-- Текст отзыва -->
                            <div class="bg-gradient-to-r from-amber-50 to-blue-50 rounded-lg p-4 border-l-4 border-blue-400">
                                <p class="text-gray-800 leading-relaxed">
                                    <?php echo htmlspecialchars($text); ?>
                                </p>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </div>
    </div>

    <style>
        /* Для mobile toggle кнопки если нужна */
        @media (max-width: 1024px) {
            .main-content-responsive {
                margin-left: 0 !important;
            }
        }
    </style>
</body>
</html>