<?php
// manager_my_zayavki.php - Статистика заявок для менеджера
// Обработка POST-запроса
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');
   
    $input = json_decode(file_get_contents('php://input'), true);
    $action = $input['action'] ?? '';
    $user_id = (int)($input['user_id'] ?? 0);
   
    if ($user_id <= 0) {
        echo json_encode(['success' => false, 'message' => 'Неверный user_id']);
        exit();
    }
   
    try {
        // Подключение к базе данных SQLite
        $db = new PDO('sqlite:data.db');
        $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
       
        // Проверка наличия user_id в таблице manager
        if ($action === 'check') {
            $stmt = $db->prepare("SELECT user_id FROM manager WHERE user_id = ?");
            $stmt->execute([$user_id]);
            $exists = $stmt->fetch() !== false;
           
            echo json_encode(['success' => $exists, 'message' => $exists ? 'Успешно' : 'Вам отказано в доступе']);
            exit();
        }
       
        // Получение статистики
        if ($action === 'get_stats') {
            $stats = [
                'total_processed' => 0,
                'total_amount' => 0.0,
                'sim_cards' => 0,
                'qr_codes' => 0,
                'cards' => 0
            ];
           
            // Общее количество обработанных заявок (статус = 'true' или 'complete' и manager совпадает)
            $stmt = $db->prepare("SELECT COUNT(*) as cnt FROM payments WHERE manager = ? AND (status = 'true' OR status = 'complete')");
            $stmt->execute([$user_id]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $stats['total_processed'] = (int) ($result['cnt'] ?? 0);
           
            // Общая сумма обработанных заявок
            $stmt = $db->prepare("SELECT COALESCE(SUM(summa), 0) as total FROM payments WHERE manager = ? AND (status = 'true' OR status = 'complete')");
            $stmt->execute([$user_id]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $stats['total_amount'] = (float) ($result['total'] ?? 0.0);
           
            // Сим-карты: есть значение в поле phone
            $stmt = $db->prepare("SELECT COUNT(*) as cnt FROM payments WHERE manager = ? AND (status = 'true' OR status = 'complete') AND phone IS NOT NULL AND phone != ''");
            $stmt->execute([$user_id]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $stats['sim_cards'] = (int) ($result['cnt'] ?? 0);
           
            // QR-коды: есть значения в полях qr_code И link
            $stmt = $db->prepare("SELECT COUNT(*) as cnt FROM payments WHERE manager = ? AND (status = 'true' OR status = 'complete') AND qr_code IS NOT NULL AND qr_code != '' AND link IS NOT NULL AND link != ''");
            $stmt->execute([$user_id]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $stats['qr_codes'] = (int) ($result['cnt'] ?? 0);
           
            // Карты: есть значения в полях card_number, card_date И cvv
            $stmt = $db->prepare("SELECT COUNT(*) as cnt FROM payments WHERE manager = ? AND (status = 'true' OR status = 'complete') AND card_number IS NOT NULL AND card_number != '' AND card_date IS NOT NULL AND card_date != '' AND cvv IS NOT NULL AND cvv != ''");
            $stmt->execute([$user_id]);
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            $stats['cards'] = (int) ($result['cnt'] ?? 0);
           
            echo json_encode(['success' => true, 'stats' => $stats]);
            exit();
        }
    } catch (PDOException $e) {
        echo json_encode(['success' => false, 'message' => 'Ошибка базы данных: ' . $e->getMessage()]);
        exit();
    }
}
?>
<!DOCTYPE html>
<html lang="ru">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Моя статистика - Панель менеджера</title>
    <!-- Telegram WebApp Script -->
    <script src="https://telegram.org/js/telegram-web-app.js?59"></script>
    <!-- Tailwind CSS CDN для стиля -->
    <script src="https://cdn.tailwindcss.com"></script>
    <style>
        body { background: linear-gradient(135deg, #f8fafc 0%, #e2e8f0 100%); }
        .menu-button {
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        }
        .menu-button:hover {
            transform: translateY(-2px);
            box-shadow: 0 10px 25px -3px rgba(0, 0, 0, 0.1), 0 4px 6px -2px rgba(0, 0, 0, 0.05);
        }
        .stat-card {
            background: rgba(255, 255, 255, 0.8);
            backdrop-filter: blur(16px);
            border: 1px solid rgba(255, 255, 255, 0.2);
        }
    </style>
</head>
<body class="min-h-screen font-sans antialiased">
    <div id="app" class="w-full max-w-md p-4 mx-auto">
        <div id="loading" class="text-center">
            <div class="animate-spin rounded-full h-12 w-12 border-b-2 border-purple-500 mx-auto mb-4"></div>
            <p class="text-gray-600">Проверка доступа...</p>
        </div>
       
        <div id="success" class="hidden">
            <div class="text-center mb-6">
                <h1 class="text-2xl font-bold text-gray-800">Моя статистика</h1>
            </div>
            <div id="stats-container">
                <!-- Stats will be loaded here -->
            </div>
            <div class="mt-6 space-y-2">
                <a href="manager_zayavki.php" class="block menu-button bg-gradient-to-r from-gray-500 to-gray-600 text-white px-6 py-3 rounded-xl font-medium text-center no-underline">
                    Назад
                </a>
                <button class="w-full menu-button bg-gradient-to-r from-red-500 to-red-600 text-white px-6 py-3 rounded-xl font-medium" onclick="closeApp()">
                    ❌ Закрыть
                </button>
            </div>
        </div>
       
        <div id="error" class="hidden text-center">
            <div class="text-6xl mb-4">❌</div>
            <h1 class="text-2xl font-bold text-red-600 mb-2">Вам отказано в доступе</h1>
            <p class="text-gray-600">Обратитесь к администратору.</p>
        </div>
       
        <div id="telegram-only" class="hidden text-center">
            <div class="text-6xl mb-4">📱</div>
            <h1 class="text-2xl font-bold text-gray-600 mb-2">Приложение работает только в Telegram</h1>
            <p class="text-gray-500">Откройте через бота в Telegram.</p>
        </div>
    </div>
    <script>
        // Инициализация Telegram WebApp
        let tg = window.Telegram?.WebApp;
        let currentUserId = null;
        if (tg) {
            tg.ready();
            tg.expand();
            currentUserId = tg.initDataUnsafe?.user?.id;
        }
        async function checkAccess() {
            if (!tg) {
                showTelegramOnly();
                return;
            }
            if (!currentUserId) {
                showError('Не удалось получить данные пользователя');
                return;
            }
            try {
                const response = await fetch('', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ action: 'check', user_id: currentUserId })
                });
                const data = await response.json();
                if (data.success) {
                    showSuccess(data.message);
                    loadStats();
                } else {
                    showError(data.message);
                }
            } catch (error) {
                console.error('Fetch error:', error);
                showError('Ошибка проверки: ' + error.message);
            }
        }
        function showSuccess(message) {
            document.getElementById('loading').classList.add('hidden');
            document.getElementById('success').classList.remove('hidden');
        }
        function showError(message) {
            document.getElementById('loading').classList.add('hidden');
            document.getElementById('error').classList.remove('hidden');
            tg?.showAlert(message);
        }
        function showTelegramOnly() {
            document.getElementById('loading').classList.add('hidden');
            document.getElementById('telegram-only').classList.remove('hidden');
        }
        // Загрузка статистики
        async function loadStats() {
            if (!currentUserId) return;
            try {
                const response = await fetch('', {
                    method: 'POST',
                    headers: { 'Content-Type': 'application/json' },
                    body: JSON.stringify({ action: 'get_stats', user_id: currentUserId })
                });
                const data = await response.json();
                if (data.success && data.stats) {
                    renderStats(data.stats);
                } else {
                    console.error('Stats error:', data);
                    renderEmptyStats();
                }
            } catch (error) {
                console.error('Load stats error:', error);
                renderEmptyStats();
            }
        }
        function renderStats(stats) {
            const html = `
                <div class="space-y-4">
                    <div class="stat-card rounded-2xl p-4 text-center">
                        <h2 class="text-lg font-semibold text-gray-700 mb-2">Обработано заявок</h2>
                        <p class="text-3xl font-bold text-blue-600">${stats.total_processed}</p>
                    </div>
                    <div class="stat-card rounded-2xl p-4 text-center">
                        <h2 class="text-lg font-semibold text-gray-700 mb-2">Общая сумма</h2>
                        <p class="text-3xl font-bold text-green-600">${new Intl.NumberFormat('ru-RU').format(Math.round(stats.total_amount))} ₽</p>
                    </div>
                    <div class="grid grid-cols-1 gap-4">
                        <div class="stat-card rounded-xl p-3 text-center">
                            <h3 class="text-sm font-medium text-gray-600 mb-1">Сим-карт</h3>
                            <p class="text-xl font-bold text-blue-600">${stats.sim_cards}</p>
                        </div>
                        <div class="stat-card rounded-xl p-3 text-center">
                            <h3 class="text-sm font-medium text-gray-600 mb-1">QR-кодов</h3>
                            <p class="text-xl font-bold text-purple-600">${stats.qr_codes}</p>
                        </div>
                        <div class="stat-card rounded-xl p-3 text-center">
                            <h3 class="text-sm font-medium text-gray-600 mb-1">Карт</h3>
                            <p class="text-xl font-bold text-green-600">${stats.cards}</p>
                        </div>
                    </div>
                </div>
            `;
            document.getElementById('stats-container').innerHTML = html;
        }
        function renderEmptyStats() {
            document.getElementById('stats-container').innerHTML = `
                <div class="text-center text-gray-500 py-8">
                    <div class="text-4xl mb-2">📊</div>
                    <p class="text-lg">Статистика недоступна</p>
                </div>
            `;
        }
        // Функция закрытия WebApp
        function closeApp() {
            tg?.close();
        }
        // Запуск проверки при загрузке
        window.addEventListener('load', checkAccess);
    </script>
</body>
</html>