# handlers/zayavki.py
from aiogram import Router, F
from aiogram.filters import StateFilter
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from aiogram.types import Message, InlineKeyboardMarkup, InlineKeyboardButton, CallbackQuery
import sqlite3
import logging

# Настройка логирования в файл log.txt
logging.basicConfig(filename='log.txt', level=logging.ERROR, format='%(asctime)s - %(levelname)s - %(message)s')

# Импорт клавиатуры
from keyboard import get_main_menu_keyboard

router = Router()

class ZayavkiStates(StatesGroup):
    choosing_type = State()

def get_user_payments(user_id: int):
    """Получение всех заявок на оплату пользователя из БД"""
    try:
        conn = sqlite3.connect('data.db')
        cursor = conn.cursor()
        cursor.execute('''
            SELECT id_zayavki, status, summa, date_time, phone FROM payments
            WHERE user_id = ?
            ORDER BY date_time DESC
        ''', (user_id,))
        payments = cursor.fetchall()
        conn.close()
        return payments
    except Exception as e:
        logging.error(f"Ошибка получения заявок на оплату: {str(e)}")
        return []

def get_user_withdrawals(user_id: int):
    """Получение всех заявок на вывод пользователя из БД"""
    try:
        conn = sqlite3.connect('data.db')
        cursor = conn.cursor()
        cursor.execute('''
            SELECT id_zayavki, status, summa, date_time FROM withdrawals
            WHERE user_id = ?
            ORDER BY date_time DESC
        ''', (user_id,))
        withdrawals = cursor.fetchall()
        conn.close()
        return withdrawals
    except Exception as e:
        logging.error(f"Ошибка получения заявок на вывод: {str(e)}")
        return []

def format_status(status: str) -> tuple:
    """Форматирование статуса с эмодзи"""
    status_display = {
        'load': {'emoji': '⏳', 'text': 'В обработке'},
        'true': {'emoji': '✅', 'text': 'Завершено'},
        'complete': {'emoji': '✅', 'text': 'Завершено'},
        'false': {'emoji': '❌', 'text': 'Отклонено'},
        'pending': {'emoji': '⏳', 'text': 'В обработке'},
        'sms_load': {'emoji': '⏳', 'text': 'В обработке'}
    }
    display_info = status_display.get(status, {'emoji': '❓', 'text': 'Неизвестно'})
    
    # Если статус содержит "true" или "!", то это завершено
    if 'true' in str(status).lower() or '!' in str(status):
        return '✅', 'Завершено'
    
    return display_info['emoji'], display_info['text']

@router.message(F.text == "📋 Мои заявки")
async def my_zayavki_handler(message: Message, state: FSMContext):
    """Выбор типа заявок"""
    await state.set_state(ZayavkiStates.choosing_type)
    
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="💸 Заявки на вывод", callback_data="zayavki_withdrawal")],
        [InlineKeyboardButton(text="💰 Заявки на оплату", callback_data="zayavki_payment")],
        [InlineKeyboardButton(text="🔙 Назад", callback_data="back_to_menu")]
    ])
    
    await message.answer("Выберите тип заявок:", reply_markup=keyboard)

@router.callback_query(F.data == "zayavki_withdrawal", StateFilter(ZayavkiStates.choosing_type))
async def show_withdrawals(callback_query: CallbackQuery, state: FSMContext):
    """Вывод заявок на вывод средств"""
    await callback_query.answer()
    
    user_id = callback_query.from_user.id
    withdrawals = get_user_withdrawals(user_id)
    
    if not withdrawals:
        text = "❌ У вас нет заявок на вывод."
    else:
        text = "💸 Ваши заявки на вывод:\n\n"
        
        for i, (id_zayavki, status, summa, date_time) in enumerate(withdrawals, 1):
            status_emoji, status_text = format_status(status)
            
            text += f"{i}. ID: {id_zayavki}\n"
            text += f"   Статус: {status_emoji} {status_text}\n"
            text += f"   Сумма: {summa:.2f} рублей\n"
            text += f"   Дата: {date_time}\n\n"
    
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="🔙 Назад", callback_data="back_to_zayavki_menu")]
    ])
    
    await callback_query.message.answer(text, reply_markup=keyboard)
    await state.clear()

@router.callback_query(F.data == "zayavki_payment", StateFilter(ZayavkiStates.choosing_type))
async def show_payments(callback_query: CallbackQuery, state: FSMContext):
    """Вывод заявок на оплату"""
    await callback_query.answer()
    
    user_id = callback_query.from_user.id
    payments = get_user_payments(user_id)
    
    if not payments:
        text = "❌ У вас нет заявок на оплату."
    else:
        text = "💰 Ваши заявки на оплату:\n\n"
        
        for i, payment in enumerate(payments, 1):
            id_zayavki = payment[0]
            status = payment[1]
            summa = payment[2]
            date_time = payment[3]
            phone = payment[4] if len(payment) > 4 else "N/A"
            
            status_emoji, status_text = format_status(status)
            
            text += f"{i}. ID: {id_zayavki}\n"
            text += f"   Статус: {status_emoji} {status_text}\n"
            text += f"   Сумма: {summa:.2f} рублей\n"
            text += f"   Номер: {phone}\n"
            text += f"   Дата: {date_time}\n\n"
    
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="🔙 Назад", callback_data="back_to_zayavki_menu")]
    ])
    
    await callback_query.message.answer(text, reply_markup=keyboard)
    await state.clear()

@router.callback_query(F.data == "back_to_zayavki_menu")
async def back_to_zayavki_menu(callback_query: CallbackQuery, state: FSMContext):
    """Возврат к меню выбора заявок"""
    await callback_query.answer()
    
    keyboard = InlineKeyboardMarkup(inline_keyboard=[
        [InlineKeyboardButton(text="💸 Заявки на вывод", callback_data="zayavki_withdrawal")],
        [InlineKeyboardButton(text="💰 Заявки на оплату", callback_data="zayavki_payment")],
        [InlineKeyboardButton(text="🔙 Назад", callback_data="back_to_menu")]
    ])
    
    await callback_query.message.answer("Выберите тип заявок:", reply_markup=keyboard)

@router.callback_query(F.data == "back_to_menu")
async def back_to_menu_handler(callback_query: CallbackQuery, state: FSMContext):
    """Возврат в главное меню"""
    await callback_query.answer()
    keyboard = get_main_menu_keyboard()
    await callback_query.message.answer("Главное меню:", reply_markup=keyboard)
    await state.clear()