# handlers/support.py
from aiogram import Router, F
from aiogram.filters import StateFilter
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from aiogram.types import Message, ReplyKeyboardRemove, ReplyKeyboardMarkup, KeyboardButton, InlineKeyboardMarkup, InlineKeyboardButton, CallbackQuery
import logging
import sqlite3 # Для хранения thread_id по user_id
from datetime import datetime
import json
import os
# Настройка логирования
logging.basicConfig(filename='log.txt', level=logging.ERROR, format='%(asctime)s - %(levelname)s - %(message)s')
from keyboard import get_main_menu_keyboard
router = Router()
def get_chat_id(key: str):
    """Загрузка chat_id из chats.json"""
    try:
        with open(os.path.join(os.path.dirname(os.path.dirname(__file__)), 'chats.json'), 'r', encoding='utf-8') as f:
            data = json.load(f)
        return data.get(key)
    except Exception as e:
        logging.error(f"Ошибка загрузки chats.json: {str(e)}")
        return None
GROUP_ID = get_chat_id("support") or -1003188015111 # Fallback если не удалось загрузить
class SupportStates(StatesGroup):
    wait_message = State()
class ReplyStates(StatesGroup):
    wait_reply = State()
def get_user_thread_id(user_id: int):
    """Получить thread_id для пользователя из БД"""
    try:
        conn = sqlite3.connect('data.db')
        cursor = conn.cursor()
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS support_threads (
                user_id INTEGER PRIMARY KEY,
                thread_id INTEGER
            )
        ''')
        conn.commit()
        cursor.execute('SELECT thread_id FROM support_threads WHERE user_id = ?', (user_id,))
        result = cursor.fetchone()
        conn.close()
        return result[0] if result else None
    except Exception as e:
        logging.error(f"Ошибка получения thread_id: {str(e)}")
        return None
def save_user_thread_id(user_id: int, thread_id: int):
    """Сохранить thread_id для пользователя"""
    try:
        conn = sqlite3.connect('data.db')
        cursor = conn.cursor()
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS support_threads (
                user_id INTEGER PRIMARY KEY,
                thread_id INTEGER
            )
        ''')
        cursor.execute('''
            INSERT OR REPLACE INTO support_threads (user_id, thread_id) VALUES (?, ?)
        ''', (user_id, thread_id))
        conn.commit()
        conn.close()
    except Exception as e:
        logging.error(f"Ошибка сохранения thread_id: {str(e)}")
@router.message(F.text == "❓ Тех поддержка")
async def support_start(message: Message, state: FSMContext):
    user_id = message.from_user.id
  
    text = "📞 Техническая поддержка\n\n"
    text += "Напишите вашу проблему или вопрос, и мы постараемся вам помочь."
  
    # Клавиатура с кнопкой Отмена
    keyboard = ReplyKeyboardMarkup(
        keyboard=[[KeyboardButton(text="❌ Отмена")]],
        resize_keyboard=True,
        one_time_keyboard=True
    )
  
    await message.answer(text, reply_markup=keyboard)
    await state.set_state(SupportStates.wait_message)
@router.message(StateFilter(SupportStates.wait_message))
async def process_support_message(message: Message, state: FSMContext):
    user_id = message.from_user.id
    user_message = message.text
    date_time = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
  
    if user_message == "❌ Отмена":
        keyboard = get_main_menu_keyboard()
        await message.answer("Главное меню:", reply_markup=keyboard)
        await state.clear()
        return
  
    # Получаем информацию о пользователе
    username = message.from_user.username or "без username"
    first_name = message.from_user.first_name or ""
    last_name = message.from_user.last_name or ""
    full_name = f"{first_name} {last_name}".strip() or "без имени"
  
    # Получаем или создаём thread_id (только если не существует)
    thread_id = get_user_thread_id(user_id)
    if not thread_id:
        try:
            # Создаём новый топик только с именем пользователя
            created_topic = await message.bot.create_forum_topic(
                chat_id=GROUP_ID,
                name=full_name
            )
            thread_id = created_topic.message_thread_id
            save_user_thread_id(user_id, thread_id)
            logging.info(f"Создан топик {thread_id} для user_id {user_id}")
        except Exception as e:
            logging.error(f"Ошибка создания топика: {str(e)}")
            # Fallback: отправляем в общий чат
            thread_id = None
  
    try:
        # Отправляем короткое сообщение в топик: текст пользователя + кнопка Ответить
        keyboard = InlineKeyboardMarkup(inline_keyboard=[
            [InlineKeyboardButton(text="Ответить", callback_data=f"reply_to_{user_id}")]
        ])
      
        await message.bot.send_message(
            chat_id=GROUP_ID,
            text=user_message,
            message_thread_id=thread_id, # Если None — в общий чат
            reply_markup=keyboard
        )
      
        await message.answer(
            "✅ Ваше сообщение отправлено в поддержку.\n"
            "Мы ответим вам в ближайшее время."
        )
      
        logging.info(f"Сообщение поддержки от {user_id}: {user_message}")
      
    except Exception as e:
        logging.error(f"Ошибка отправки сообщения поддержки: {str(e)}")
        await message.answer(
            "❌ Ошибка при отправке сообщения. Попробуйте позже."
        )
  
    # Возврат в главное меню
    keyboard = get_main_menu_keyboard()
    await message.answer("Главное меню:", reply_markup=keyboard)
  
    await state.clear()
@router.callback_query(F.data.startswith("reply_to_"))
async def handle_reply_button(callback: CallbackQuery, state: FSMContext):
    user_id_str = callback.data.split("_")[-1]
    try:
        target_user_id = int(user_id_str)
    except ValueError:
        await callback.answer("Ошибка: неверный ID пользователя.")
        return
  
    # Удаляем клавиатуру с кнопкой
    await callback.message.edit_reply_markup(reply_markup=None)
  
    await callback.message.answer(
        f"Напишите ответ пользователю (ID: {target_user_id}).\n"
        "Для отмены введите /cancel."
    )
  
    await state.set_state(ReplyStates.wait_reply)
    await state.update_data(target_user_id=target_user_id)
    await callback.answer()
@router.message(StateFilter(ReplyStates.wait_reply))
async def process_reply_message(message: Message, state: FSMContext):
    if message.text == "/cancel":
        await message.answer("Ответ отменен.")
        await state.clear()
        return
  
    data = await state.get_data()
    target_user_id = data.get("target_user_id")
    if not target_user_id:
        await message.answer("Ошибка: нет данных о пользователе.")
        await state.clear()
        return
  
    if message.chat.id != GROUP_ID:
        await message.answer("Этот режим работает только в группе поддержки.")
        await state.clear()
        return
  
    try:
        # Отправляем ответ пользователю
        await message.bot.send_message(
            chat_id=target_user_id,
            text=f"💬 Ответ от поддержки:\n\n{message.text}"
        )
      
        await message.answer("✅ Ответ отправлен пользователю.")
        logging.info(f"Ответ от {message.from_user.id} пользователю {target_user_id}: {message.text}")
      
    except Exception as e:
        logging.error(f"Ошибка отправки ответа пользователю {target_user_id}: {str(e)}")
        await message.answer("❌ Ошибка при отправке ответа.")
  
    await state.clear()