# handlers/sim_card.py
from aiogram import Router, F
from aiogram.filters import StateFilter
from aiogram.fsm.context import FSMContext
from aiogram.fsm.state import State, StatesGroup
from aiogram.types import Message, InlineKeyboardMarkup, InlineKeyboardButton
import random
from datetime import datetime
import logging
import json
import os
from db import get_user_balance, update_user_balance, save_payment # Импорт для баланса и сохранения заявки
from keyboard import get_main_menu_keyboard # Для возврата, если нужно
# Настройка логирования в файл log.txt
logging.basicConfig(filename='log.txt', level=logging.ERROR, format='%(asctime)s - %(levelname)s - %(message)s')
router = Router()
class SimCardStates(StatesGroup):
    wait_phone = State()
    wait_amount = State()
def get_chat_id(key: str):
    """Загрузка chat_id из chats.json"""
    try:
        with open(os.path.join(os.path.dirname(os.path.dirname(__file__)), 'chats.json'), 'r', encoding='utf-8') as f:
            data = json.load(f)
        return data.get(key)
    except Exception as e:
        logging.error(f"Ошибка загрузки chats.json: {str(e)}")
        return None
def get_function_status(key: str):
    """Загрузка статуса функции из function.json"""
    try:
        with open(os.path.join(os.path.dirname(os.path.dirname(__file__)), 'function.json'), 'r', encoding='utf-8') as f:
            data = json.load(f)
        return data.get(key, True)
    except Exception as e:
        logging.error(f"Ошибка загрузки function.json: {str(e)}")
        return True
def get_procent_rate():
    """Загрузка процента комиссии из procent.json"""
    try:
        with open(os.path.join(os.path.dirname(os.path.dirname(__file__)), 'procent.json'), 'r', encoding='utf-8') as f:
            data = json.load(f)
        return data.get('procent', 10) / 100  # Возвращаем как десятичную дробь
    except Exception as e:
        logging.error(f"Ошибка загрузки procent.json: {str(e)}")
        return 0.1  # Дефолт 10%
@router.message(F.text == "📱 Сим-Карта")
async def sim_card_start(message: Message, state: FSMContext):
    if not get_function_status('simcard'):
        await message.answer("Проводятся технические работы")
        return
    user_id = message.from_user.id
    balance = get_user_balance(user_id)
 
    if balance < 10:
        text = "❌ Ваш баланс ниже 10 рублей. Пополните свой баланс для продолжения."
        keyboard = InlineKeyboardMarkup(inline_keyboard=[
            [InlineKeyboardButton(text="💰 Пополнить баланс", callback_data="top_up_balance")],
            [InlineKeyboardButton(text="🔙 Назад", callback_data="back_to_menu")]
        ])
        await message.answer(text, reply_markup=keyboard)
        return
 
    await message.answer("Введите номер телефона для пополнения (начинающийся на +7):")
    await state.set_state(SimCardStates.wait_phone)
@router.message(StateFilter(SimCardStates.wait_phone))
async def process_phone(message: Message, state: FSMContext):
    phone = message.text.strip()
 
    if not phone.startswith('+7') or len(phone) != 12 or not phone[1:].isdigit():
        await message.answer("❌ Неверный формат номера. Введите номер, начинающийся на +7 (например, +79123456789):")
        return
 
    await state.update_data(phone=phone)
    await message.answer(f"Номер {phone} подтвержден. Теперь введите сумму пополнения:")
    await state.set_state(SimCardStates.wait_amount)
@router.message(StateFilter(SimCardStates.wait_amount))
async def process_amount(message: Message, state: FSMContext):
    try:
        amount = float(message.text)
        if amount <= 0:
            await message.answer("❌ Сумма должна быть положительной. Введите заново:")
            return
    except ValueError:
        await message.answer("❌ Неверный формат. Введите число:")
        return
 
    user_id = message.from_user.id
    balance = get_user_balance(user_id)
 
    # Получаем актуальный процент комиссии
    procent_rate = get_procent_rate()
 
    # Расчёт максимальной суммы: balance / (1 + procent_rate)
    max_amount = balance / (1 + procent_rate)
 
    if amount > max_amount:
        await message.answer(f"❌ Вы можете ввести сумму максимум {max_amount:.2f} рублей (учитывая комиссию {int(procent_rate*100)}%). Введите заново:")
        return
 
    # Списание с баланса
    commission = amount * procent_rate
    total_deduct = amount + commission
    update_user_balance(user_id, -total_deduct) # Списываем с баланса
 
    # Получение данных из state
    data = await state.get_data()
    phone = data.get('phone')
 
    # Генерация ID заявки и datetime
    date_time = datetime.now().strftime('%Y-%m-%d %H:%M:%S')
    id_zayavki = random.randint(100000, 999999)
 
    # Сохранение заявки в БД с обработкой ошибки
    try:
        save_payment(user_id, id_zayavki, "load", date_time, phone, amount)
    except Exception as e:
        logging.error(f"Ошибка в sim_card.py при сохранении заявки: {str(e)}")
        await message.answer("❌ Ошибка при сохранении заявки. Обратитесь в поддержку.")
        await state.clear()
        return
 
    # Уведомление в Telegram с обработкой ошибки
    try:
        zayavki_chat_id = get_chat_id("zayavki")
        if zayavki_chat_id:
            notification_text = f"Новая заявка\nUser ID: {user_id}\nID заявки: {id_zayavki}\nPhone: {phone}\nSumma: {amount}\nDate: {date_time}"
            await message.bot.send_message(chat_id=zayavki_chat_id, text=notification_text)
        else:
            logging.error("Не удалось получить chat_id для zayavki")
    except Exception as e:
        logging.error(f"Ошибка отправки уведомления: {str(e)}")
        # Продолжаем, даже если уведомление не ушло
 
    await message.answer(
        f"✅ Пополнение на {amount:.2f} рублей ({commission:.2f} комиссия, итого {total_deduct:.2f}) "
        f"на номер {phone} успешно!\n"
        f"Новый баланс: {get_user_balance(user_id):.2f} рублей."
    )
 
    # Возврат в главное меню
    keyboard = get_main_menu_keyboard()
    await message.answer("Операция завершена. Главное меню:", reply_markup=keyboard)
 
    await state.clear()